package com.tencent.start.cgs.tools;

import com.emc.ecs.nfsclient.nfs.NfsWriteRequest;
import com.emc.ecs.nfsclient.nfs.io.Nfs3File;
import com.emc.ecs.nfsclient.nfs.io.NfsFileInputStream;
import com.emc.ecs.nfsclient.nfs.io.NfsFileOutputStream;
import com.emc.ecs.nfsclient.nfs.nfs3.Nfs3;
import com.emc.ecs.nfsclient.rpc.CredentialUnix;

import java.io.*;
import java.nio.file.Files;
import java.util.Iterator;
import java.util.List;

import static com.tencent.start.cgs.tools.RandomAccessOutputFile.createNewFile;

@SuppressWarnings("unused")
public abstract class AbstractFile {

    protected final String url;

    protected AbstractFile(String url) {
        this.url = url;
    }

    public abstract String getAbsolutePath();

    public abstract String getName();

    public abstract String getParent();

    public abstract AbstractFile getParentFile() throws IOException;

    public abstract AbstractFile getChildFile(String name) throws IOException;

    public abstract long length() throws IOException;

    public abstract long lastModified() throws IOException;

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    public abstract boolean renameTo(AbstractFile dest);

    public abstract boolean mkdirs() throws IOException;

    public abstract boolean exists() throws IOException;

    public abstract boolean delete() throws IOException;

    public abstract boolean isDirectory() throws IOException;

    public abstract boolean isFile() throws IOException;

    public abstract AbstractFile[] listFiles() throws IOException;

    public abstract InputStream getInputStream(long offset) throws IOException;

    public abstract OutputStream getOutputStream(long offset) throws IOException;

    public abstract Object getClassType();

    public abstract Object asFile();

    public abstract void link(AbstractFile source) throws IOException;

    public static AbstractFile make(String url) throws IOException {
        if (url.startsWith("nfs:")) {
            return new NfsAbstractFile(url);
        } else {
            return new NativeAbstractFile(url);
        }
    }

    public static AbstractFile make(File file) {
        return new NativeAbstractFile(file);
    }

    public static AbstractFile make(Nfs3File file) {
        return new NfsAbstractFile(file);
    }

    static class NfsAbstractFile extends AbstractFile {

        final Nfs3File file;

        // nfs://server/export/path -> server:/export/path
        public static Nfs3File getNfs3File(String url) throws IOException {
            if (File.separatorChar != '/') {
                url = url.replace(File.separatorChar, '/');
            }
            if (!url.startsWith("nfs://")) {
                throw new IOException("invalid nfs file: " + url);
            }
            int n = url.indexOf('/', 6);
            if (-1 == n) {
                throw new IOException("invalid nfs file: " + url);
            }
            int m = url.indexOf('/', n + 1);
            if (-1 == m) {
                throw new IOException("invalid nfs file: " + url);
            }
            String server = url.substring(6, n);
            String export = url.substring(n, m);
            String path = url.substring(m);
            //System.out.println("server=" + server + ", export=" + export + ", path=" + path);
            return new Nfs3File(new Nfs3(server, export,
                    new CredentialUnix(0, 0, null), 3), path);
        }

        NfsAbstractFile(String url) throws IOException {
            super(url);
            file = getNfs3File(url);
        }

        NfsAbstractFile(Nfs3File file) {
            super(App.pathJoin("nfs://" + file.getNfs().getServer(),
                    file.getNfs().getExportedPath(), file.getPath()));
            this.file = file;
        }

        @Override
        public String getAbsolutePath() {
            return url;
        }

        @Override
        public String getName() {
            return file.getName();
        }

        @Override
        public String getParent() {
            return file.getParent();
        }

        @Override
        public AbstractFile getParentFile() throws IOException {
            return new NfsAbstractFile(new Nfs3File(file.getNfs(), file.getParent()));
        }

        @Override
        public AbstractFile getChildFile(String name) throws IOException {
            return new NfsAbstractFile(new Nfs3File(file, name));
        }

        @Override
        public long length() throws IOException {
            return file.length();
        }

        public long lastModified() throws IOException {
            return file.lastModified();
        }

        @Override
        public boolean renameTo(AbstractFile dest) {
            try {
                return file.renameTo(((NfsAbstractFile) dest).file);
            } catch (IOException e) {
                return false;
            }
        }

        @Override
        public boolean mkdirs() {
            try {
                file.mkdirs();
                return true;
            } catch (IOException e) {
                return false;
            }
        }

        @Override
        public boolean exists() throws IOException {
            return file.exists();
        }

        @Override
        public boolean delete() throws IOException {
            file.delete();
            return true;
        }

        @Override
        public boolean isDirectory() throws IOException {
            return file.isDirectory();
        }

        @Override
        public boolean isFile() throws IOException {
            return file.isFile();
        }

        @Override
        public AbstractFile[] listFiles() throws IOException {
            List<Nfs3File> files = file.listFiles();
            if (null == files) {
                return null;
            }
            Iterator<Nfs3File> it = files.listIterator();
            AbstractFile[] result = new AbstractFile[files.size()];
            for (int i = 0; i < files.size(); ++i) {
                result[i] = new NfsAbstractFile(it.next());
            }
            return result;
        }

        @Override
        public InputStream getInputStream(long offset) throws IOException {
            return new NfsFileInputStream(file, offset, Integer.MAX_VALUE);
        }

        @Override
        public OutputStream getOutputStream(long offset) throws IOException {
            if (!file.exists()) {
                createNewFile(file);
            }
            return new NfsFileOutputStream(file, offset, NfsWriteRequest.DATA_SYNC);
        }

        @Override
        public Object getClassType() {
            return Nfs3File.class;
        }

        @Override
        public Object asFile() {
            return file;
        }

        @Override
        public void link(AbstractFile source) throws IOException {
            if (!source.getClassType().equals(Nfs3File.class)) {
                throw new IOException("source must be a instance of Nfs3File");
            }
            file.link((Nfs3File) source.asFile());
        }

        @Override
        public String toString() {
            return file.toString();
        }
    }

    static class NativeAbstractFile extends AbstractFile {

        final File file;
        String name;

        NativeAbstractFile(String path) {
            super(path);
            file = new File(path).getAbsoluteFile();
        }

        NativeAbstractFile(File file) {
            super(file.getPath());
            this.file = file;
        }

        @Override
        public String getAbsolutePath() {
            return file.getAbsolutePath();
        }

        @Override
        public String getName() {
            return name != null ? name : (name = file.getName());
        }

        @Override
        public String getParent() {
            return file.getParent();
        }

        @Override
        public AbstractFile getParentFile() {
            return new NativeAbstractFile(file.getParentFile());
        }

        @Override
        public AbstractFile getChildFile(String name) {
            return new NativeAbstractFile(new File(file, name));
        }

        @Override
        public long length() throws IOException {
            return file.length();
        }

        public long lastModified() throws IOException {
            return file.lastModified();
        }

        @Override
        public boolean renameTo(AbstractFile dest) {
            return file.renameTo(((NativeAbstractFile) dest).file);
        }

        @Override
        public boolean mkdirs() {
            return file.mkdirs();
        }

        @Override
        public boolean exists() {
            return file.exists();
        }

        @Override
        public boolean delete() {
            return file.delete();
        }

        @Override
        public boolean isDirectory() {
            return file.isDirectory();
        }

        @Override
        public boolean isFile() {
            return file.isFile();
        }

        @Override
        public AbstractFile[] listFiles() {
            File[] files = file.listFiles();
            if (null == files) {
                return null;
            }
            AbstractFile[] result = new AbstractFile[files.length];
            for (int i = 0; i < files.length; ++i) {
                result[i] = new NativeAbstractFile(files[i]);
            }
            return result;
        }

        @Override
        @SuppressWarnings("all")
        public InputStream getInputStream(long offset) throws IOException {
            FileInputStream in = new FileInputStream(file);
            if (0 != offset) {
                in.getChannel().position(offset);
            }
            return in;
        }

        @Override
        @SuppressWarnings("all")
        public OutputStream getOutputStream(long offset) throws IOException {
            FileOutputStream out = new FileOutputStream(file);
            if (0 != offset) {
                out.getChannel().position(offset);
            }
            return out;
        }

        @Override
        public Object getClassType() {
            return File.class;
        }

        @Override
        public Object asFile() {
            return file;
        }

        @Override
        public void link(AbstractFile source) throws IOException {
            Files.createLink(file.toPath(), ((File) source.asFile()).toPath());
        }

        @Override
        public String toString() {
            return file.toString();
        }
    }
}
