package com.tencent.start.cgs.tools;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.tencent.start.cgs.tools.ZpkFile.*;
import static com.tencent.start.cgs.tools.ZpkMetadata.FileNode;

public class ZpkFileDiff extends ZpkFilePack {
    private static final String ZDIF_EXT_NAME = ".zdif";
    private static final String ZDIF_HEADER_ENT_NAME = ".cgs_vfs_zdif";
    private static final String ZDIF_DATA_ENT_NAME = ".cgs_vfs_zdif_data";
    private static final String ZDIF_IDX_ENT_NAME = ".cgs_vfs_zdif_idx";
    private static final int PART_SIZE = 65536;
    private static final Object DATA_TYPE_MATCH = Boolean.TRUE;
    private static final Object DATA_TYPE_DIFF = Boolean.FALSE;
    private ZpkFile.Finder finder;
    private ZpkFile.Version.Reader zpkReader;
    private AbstractFile outputFile;
    private RandomAccessOutputFile out;
    private final PartList partList = new PartList();
    private WritingFileInfo currentWritingFile;
    private final CRC32 crc32_2 = new CRC32();

    static class Part {
        Part next;
        public final Object type;
        public final long srcPos;
        public final long dstPos;
        public long size;
        public Part(Object type, long srcPos, long dstPos, long size) {
            this.type = type;
            this.srcPos = srcPos;
            this.dstPos = dstPos;
            this.size = size;
            this.next = null;
        }
    }

    static class PartList {
        Part head, tail;
        int size = 0;

        Part pop() {
            Part h = head;
            if (size > 0) {
                --size;
                if ((head = h.next) == null) {
                    tail = null;
                } else {
                    h.next = null;
                }
            }
            return h;
        }

        void add(Part part) {
            part.next = null;
            if (tail != null) {
                tail.next = part;
            } else {
                head = part;
            }
            tail = part;
            ++size;
        }

        void add(PartList partList) {
            if (this.tail == null) {
                this.head = partList.head;
                this.tail = partList.tail;
            } else if (partList.head != null) {
                this.tail.next = partList.head;
                this.tail = partList.tail;
            }
            size += partList.size;
        }

        Part getHead() {
            return this.head;
        }

        Part getTail() {
            return this.tail;
        }

        boolean isEmpty() {
            return 0 == this.size;
        }
    }

    static long getTotalSize(PartList l) {
        long size = 0;
        Part n = l.head;
        while (n != null) {
            size += n.size;
            n = n.next;
        }
        return size;
    }

    class WritingFileInfo implements Closeable {
        final PartList partList = new PartList();
        final FileNode fileNode;
        final FileNode matchNode;
        final RandomAccessInputFile matchFile;
        final long diffPos;
        final long dstPos;
        final int crc;

        WritingFileInfo(FileNode fileNode) throws IOException {
            this.fileNode = fileNode;
            this.matchNode = zpkReader.getNode(fileNode.relativeName);
            this.matchFile = this.matchNode != null ? zpkReader.open(this.matchNode) : null;
            this.diffPos = ZpkFileDiff.this.out.getFilePointer();
            this.dstPos = ZpkFileDiff.this.output.getFilePointer();
            this.crc = ZpkFileDiff.this.crc32_2.crc();
        }

        @Override
        public void close() throws IOException {
            if (this.matchFile != null) {
                this.matchFile.close();
            }
        }
    }

    class ZpkDiffWrapperOutputFile extends RandomAccessOutputFile {
        long pos = 0;
        long len = 0;
        final byte[] buf = new byte[1];

        @Override
        public long length() throws IOException {
            return this.len;
        }

        @Override
        public long getFilePointer() {
            return this.pos;
        }

        @Override
        public void seek(long pos) {
            this.pos = pos;
        }

        @Override
        public void setLength(long newLength) {
            this.len = newLength;
        }

        @Override
        public void write(int b) throws IOException {
            buf[0] = (byte) (b & 0x000000FF);
            write(buf, 0, 1);
        }

        @Override
        public void write(byte[] buf, int off, int len) throws IOException {
            ZpkFileDiff.this.onOutputWrite(this.pos, buf, off, len);
            this.pos += len;
            if (this.pos > this.len) {
                this.len = this.pos;
            }
        }
    }

    public ZpkFileDiff() throws IOException {
    }

    protected static void checkPosition(PartList list, long dstPos) {
        Part part = list.getTail();
        App.Assert(part == null || part.dstPos + part.size == dstPos);
    }

    protected static void addPart(PartList list, Object dataType, long srcPos, long dstPos, int size) {
        Part part = list.getTail();
        if (part != null && part.type == dataType &&
                part.dstPos + part.size == dstPos &&
                part.srcPos + part.size == srcPos) {
            part.size += size;
        } else {
            part = new Part(dataType, srcPos, dstPos, size);
            list.add(part);
        }
    }

    private void writeDif(byte[] buf, int off, int len) throws IOException {
        this.out.write(buf, off, len);
        this.crc32_2.update(buf, off, len);
    }

    protected void onOutputWrite(long dstPos, byte[] buf, final int off, final int len) throws IOException {
        App.Assert(dstPos >= 0 && off >= 0 && len > 0, "dstPos >= 0 && off >= 0 && len > 0");
        if (null == currentWritingFile || null == currentWritingFile.matchFile) {
            addPart(null == currentWritingFile ? partList : currentWritingFile.partList, DATA_TYPE_DIFF,
                    this.out.getFilePointer(), dstPos, len);
            writeDif(buf, off, len);
        } else {
            RandomAccessInputFile matchFile = currentWritingFile.matchFile;
            byte[] matchBuf = new byte[len];
            final long matchFilePos = matchFile.getFilePointer();
            final long matchPos = currentWritingFile.matchNode.dataOffset + matchFilePos;
            final int matchLen = matchFile.read(matchBuf, 0, matchBuf.length);
            int offset = 0, mismatchStartOffset = -1;
            long mismatchStartDstPos = -1;
            do {
                int mismatchIndex = offset < matchLen ? Arrays.mismatch(matchBuf, offset, matchLen,
                        buf, off + offset, off + len) : 0;
                int matchSize = mismatchIndex < 0 ? len - offset : (mismatchIndex / PART_SIZE) * PART_SIZE;
                if (matchSize > 0) {
                    if (mismatchStartOffset >= 0) {
                        final int mismatchSize = offset - mismatchStartOffset;
                        App.Assert(mismatchSize > 0, "mismatchSize > 0");
                        checkPosition(currentWritingFile.partList, mismatchStartDstPos);
                        addPart(currentWritingFile.partList, DATA_TYPE_DIFF, this.out.getFilePointer(),
                                mismatchStartDstPos, mismatchSize);
                        writeDif(buf, off + mismatchStartOffset, mismatchSize);
                        mismatchStartOffset = -1;
                    }
                    checkPosition(currentWritingFile.partList, dstPos);
                    addPart(currentWritingFile.partList, DATA_TYPE_MATCH, matchPos + offset, dstPos, matchSize);
                }
                if (mismatchStartOffset < 0) {
                    mismatchStartOffset = offset + matchSize;
                    mismatchStartDstPos = dstPos + matchSize;
                }
                offset += (matchSize + PART_SIZE);
                dstPos += (matchSize + PART_SIZE);
            } while (offset < len);
            if (mismatchStartOffset >= 0 && mismatchStartOffset < len) {
                final int mismatchSize = len - mismatchStartOffset;
                checkPosition(currentWritingFile.partList, mismatchStartDstPos);
                addPart(currentWritingFile.partList, DATA_TYPE_DIFF, this.out.getFilePointer(),
                        mismatchStartDstPos, mismatchSize);
                writeDif(buf, off + mismatchStartOffset, mismatchSize);
            }
            matchFile.seek(matchFilePos + matchLen);
        }
    }

    @Override
    protected void onWriteFileBegin(FileNode fileNode) throws IOException {
        this.currentWritingFile = new WritingFileInfo(fileNode);
    }

    @Override
    protected void onWriteFileEnd(FileNode fileNode, boolean canceled) throws IOException {
        FileNode exists;
        App.Assert(currentWritingFile.partList.isEmpty() ||
                getTotalSize(currentWritingFile.partList) == fileNode.fileSize ||
                canceled);
        if (canceled) {
            this.out.seek(currentWritingFile.diffPos);
            this.crc32_2.reset(currentWritingFile.crc);
            //System.out.println("[DBG] " + fileNode.relativeName + ", cancel write. diff pos="
            //       + humanReadableBytes(out.getFilePointer()));
        } else if ((exists = this.finder.findChecksum(fileNode)) != null) {
            this.out.seek(currentWritingFile.diffPos);
            this.crc32_2.reset(currentWritingFile.crc);
            Part tail = partList.getTail();
            if (tail != null && tail.type == DATA_TYPE_MATCH &&
                    tail.srcPos + tail.size == exists.dataOffset &&
                    tail.dstPos + tail.size == currentWritingFile.dstPos) {
                tail.size += exists.fileSize;
            } else {
                partList.add(new Part(DATA_TYPE_MATCH, exists.dataOffset, currentWritingFile.dstPos, exists.fileSize));
            }
            //System.out.println("[DBG] " + fileNode.relativeName + ", duplicated. diff pos="
            //        + humanReadableBytes(out.getFilePointer()));
        } else {
            // int num = 1;
            // Part n = currentWritingFile.partList.head;
            // while (n != null) {
            //     System.out.println("[DBG] " + fileNode.relativeName + ", part " + (num++)
            //             + ", type=" + (n.type == DATA_TYPE_MATCH ? "m" : "d") + ", dstPos="
            //             + n.dstPos + ", srcPos=" + n.srcPos + ", size=" + humanReadableBytes(n.size) + ", diff pos="
            //             + humanReadableBytes(out.getFilePointer()));
            //     n = n.next;
            // }
            Part mergeHead = currentWritingFile.partList.getHead();
            if (mergeHead != null) {
                Part tail = partList.getTail();
                if (tail.type == mergeHead.type &&
                        tail.dstPos + tail.size == mergeHead.dstPos &&
                        tail.srcPos + tail.size == mergeHead.srcPos) {
                    tail.size += mergeHead.size;
                    Part p = currentWritingFile.partList.pop();
                    UNUSED(p);
                }
                partList.add(currentWritingFile.partList);
            } else {
                App.Assert(currentWritingFile.fileNode.fileSize == 0,
                        "currentWritingFile.fileNode.fileSize == 0");
            }
        }
        currentWritingFile.close();
        currentWritingFile = null;
    }

    static final byte PATCH_OP_MATCH = 0x4d;
    static final byte PATCH_OP_DIFF = 0x44;
    static final byte PATCH_OP_SEEK = 0x53;

    public long diff(AbstractFile outputDir, AbstractFile zFile, AbstractFile inputDir,
                     ZpkMetadata.EnumFilter ef) throws IOException {
        if (null == outputDir) {
            outputDir = zFile.getParentFile();
        }
        String name = zFile.getName();
        String prefix = getZpkPrefix(name);
        String ts = getZpkTimestamp(name);
        String newName = getZpkFileName(prefix, buildTime);
        byte[] nameBytes = name.getBytes(StandardCharsets.UTF_8);
        byte[] newNameBytes = newName.getBytes(StandardCharsets.UTF_8);
        if (nameBytes.length > MAX_ZPK_FILE_NAME_LEN || newNameBytes.length > MAX_ZPK_FILE_NAME_LEN) {
            throw new IOException("file name too long: " + name + ", " + newName);
        }
        AbstractFile outputFile = outputDir.getChildFile(
                getTimestampFileName(prefix + "_" + ts + "_to", buildTime, ZDIF_EXT_NAME));
        System.out.println("Output File: " + outputFile);
        this.outputFile = outputFile;
        long bytesWritten;
        AbstractFile tempFile = getTempFile(outputFile);
        try (ZpkFile zpkFile = ZpkFile.open(zFile);
             RandomAccessOutputFile out = RandomAccessOutputFile.from(tempFile)) {
            if (buildTime < zpkFile.getLastModified()) {
                throw new IOException("time error");
            }
            this.finder = zpkFile.finder();
            this.zpkReader = zpkFile.getVersion(ts).reader();
            this.out = out;
            out.order(ByteOrder.LITTLE_ENDIAN);

            // write header section
            final int headerSize = 32 + nameBytes.length + newNameBytes.length;
            final long headerEntryPos = out.getFilePointer();
            final long headerEntrySize = ZIP_ENTRY_SIZE + ZDIF_HEADER_ENT_NAME.getBytes(StandardCharsets.UTF_8).length;
            final long headerPos = headerEntryPos + headerEntrySize;

            ByteBuffer buff = ByteBuffer.allocate(headerSize);
            buff.order(ByteOrder.LITTLE_ENDIAN);
            buff.putLong(zpkFile.length());
            buff.put(zpkFile.getChecksum());
            buff.put((byte) newNameBytes.length);
            buff.put(newNameBytes);
            buff.put((byte) nameBytes.length);
            buff.put(nameBytes);
            final byte[] headerData = buff.array();
            App.Assert(headerData.length == headerSize, "headerData.length == headerSize");
            crc32.reset();
            crc32.update(headerData, 0, headerSize);
            out.seek(headerEntryPos);
            ZpkFile.writeEntry(out, ZDIF_HEADER_ENT_NAME, headerSize, buildTime, crc32.getValue(), false);
            App.Assert(headerPos == out.getFilePointer());
            out.write(headerData, 0, headerSize);

            // write data section
            final long dataEntryPos = out.getFilePointer();
            final long dataEntrySize = ZIP_ENTRY_SIZE_64 + ZDIF_DATA_ENT_NAME.getBytes(StandardCharsets.UTF_8).length;
            final long dataPos = dataEntryPos + dataEntrySize;
            out.seek(dataPos);
            this.crc32.reset();
            this.crc32_2.reset();

            this.output = new ZpkDiffWrapperOutputFile();
            this.output.order(ByteOrder.LITTLE_ENDIAN);
            long targetPacketSize = super.build(inputDir, ef);
            this.output = null;

            final long dataEndPos  = out.getFilePointer();
            final long dataSize = dataEndPos - dataPos;

            out.seek(dataEntryPos);
            ZpkFile.writeEntry(out, ZDIF_DATA_ENT_NAME, dataSize, buildTime, this.crc32_2.getValue(), true);
            App.Assert(out.getFilePointer() == dataPos);

            // write index section
            long dstPos = 0, totalBytes = 0;
            buff = ByteBuffer.allocate(26 * partList.size);
            buff.order(ByteOrder.LITTLE_ENDIAN);
            buff.putLong(targetPacketSize);
            buff.putLong(0);
            Part next = partList.head;
            while (next != null) {
                Part part = next;
                next = part.next;
                if (part.dstPos != dstPos) {
                    buff.put(PATCH_OP_SEEK);
                    buff.putLong(part.dstPos);
                    dstPos = part.dstPos;
                }
                buff.put(part.type == DATA_TYPE_MATCH ? PATCH_OP_MATCH : PATCH_OP_DIFF);
                buff.putLong(part.srcPos);
                buff.putLong(part.size);
                dstPos += part.size;
                totalBytes += part.size;
            }
            final int opsSize = buff.position();
            buff.position(8);
            buff.putLong(totalBytes);
            buff.position(opsSize);
            out.seek(dataEndPos);
            compressEntry(out, ZDIF_IDX_ENT_NAME, buildTime, buff.array(), 0, opsSize);
            out.setLength(out.getFilePointer());
            bytesWritten = out.length();
            out.flush();

            this.out = null;
            this.zpkReader = null;
            this.finder = null;
        } catch (Exception e) {
            tempFile.delete();
            throw e;
        }
        if (outputFile.exists() && !outputFile.delete()) {
            tempFile.delete();
            throw new IOException("count not delete file: " + outputFile);
        } else if (!tempFile.renameTo(outputFile)) {
            throw new IOException("count not rename: " + outputFile);
        }
        return bytesWritten;
    }

    private static void checkByteBufferRemaining(ByteBuffer b, int size) throws IOException {
        if (b.remaining() < size) {
            throw new IOException("invalid remaining size");
        }
    }

    private static void copy(Speedometer sp, byte[] buf, RandomAccessOutputFile out, RandomAccessInputFile in,
                             final long pos, final long size) throws IOException {
        in.seek(pos);
        int bytesRead;
        long remaining = size;
        long totalBytesWritten = 0;
        while (remaining > 0 && (bytesRead = in.read(buf, 0, (int) Math.min(buf.length, remaining))) > 0) {
            out.write(buf, 0, bytesRead);
            totalBytesWritten += bytesRead;
            remaining -= bytesRead;
            sp.feed(bytesRead);
        }
        if (totalBytesWritten != size) {
            throw new IOException("totalBytesWritten != size. totalBytesWritten=" + totalBytesWritten +
                    ", size=" + size);
        }
    }

    private long patch(RandomAccessOutputFile out, RandomAccessInputFile inDiff, RandomAccessInputFile inMatch,
                       ByteBuffer ops) throws IOException {
        out.seek(0);
        ops.order(ByteOrder.LITTLE_ENDIAN);
        final long packetSize = ops.getLong();
        final long totalBytes = ops.getLong();
        final byte[] buf = new byte[4 * 1024 * 1024];
        Speedometer sp = new Speedometer(totalBytes);
        while (ops.remaining() > 0) {
            byte op = ops.get();
            switch (op) {
                case PATCH_OP_DIFF:
                case PATCH_OP_MATCH: {
                    checkByteBufferRemaining(ops, 16);
                    long srcPos = ops.getLong();
                    long size = ops.getLong();
                    try {
                        copy(sp, buf, out, op == PATCH_OP_DIFF ? inDiff : inMatch, srcPos, size);
                    } catch (Exception e) {
                        throw new IOException("copy data\n" + e + "\n" +
                                "op=" + (op == PATCH_OP_DIFF ? "PATCH_OP_DIFF" : "PATCH_OP_MATCH") + ", " +
                                "pos=" + srcPos + ", " + "size=" + size);
                    }
                    break;
                }
                case PATCH_OP_SEEK: {
                    checkByteBufferRemaining(ops, 8);
                    long pos = ops.getLong();
                    out.seek(pos);
                    break;
                }
                default:
                    throw new IOException("Invalid patch operation code: " + op);
            }
        }
        sp.finish();
        return packetSize;
    }

    public long patch(AbstractFile outputDir, AbstractFile zFile, AbstractFile zdifFile) throws IOException {
        if (null == outputDir) {
            outputDir = zFile.getParentFile();
        }
        try (RandomAccessInputFile in = RandomAccessInputFile.from(zdifFile);
             ZpkFile zpkFile = ZpkFile.open(zFile)) {
            in.order(ByteOrder.LITTLE_ENDIAN);
            ZpkEntry headEntry = readEntry(in);
            if (!ZDIF_HEADER_ENT_NAME.equals(headEntry.name) || headEntry.size < 62 || headEntry.size > 512) {
                throw new IOException("not valid diff file: " + zdifFile);
            }
            final long dataEntryPos = in.getFilePointer() + headEntry.size;
            final byte[] checksum = new byte[16];
            final long length = in.readLong();
            in.readFully(checksum);
            final int newNameLength = in.read();
            if (newNameLength > MAX_ZPK_FILE_NAME_LEN) {
                throw new IOException("invalid name length: " + newNameLength);
            }
            final String newZpkFileName = new String(in.readBytes(newNameLength), StandardCharsets.UTF_8);
            final int oldNameLength = in.read();
            if (oldNameLength > MAX_ZPK_FILE_NAME_LEN) {
                throw new IOException("invalid name length: " + oldNameLength);
            }
            final String oldZpkFileName = new String(in.readBytes(oldNameLength), StandardCharsets.UTF_8);
            if (zpkFile.length != length || !Arrays.equals(zpkFile.getChecksum(), checksum)) {
                throw new IOException("zpk file not match. expected " + oldZpkFileName);
            } else if (!oldZpkFileName.equals(zFile.getName())) {
                System.out.println("WARING: zpk file name not equals. expected \"" + oldZpkFileName +
                        "\" but get \"" + zFile.getName() + "\".");
            }

            in.seek(dataEntryPos);
            ZpkEntry dataEntry = readEntry(in);
            if (!ZDIF_DATA_ENT_NAME.equals(dataEntry.name)) {
                throw new IOException("invalid zpk diff file: " + zdifFile);
            }

            in.seek(in.getFilePointer() + dataEntry.size);
            Map.Entry<String, byte[]> indexEntry = uncompressEntry(in);
            if (!ZDIF_IDX_ENT_NAME.equals(indexEntry.getKey())) {
                throw new IOException("invalid zpk diff file: " + zdifFile);
            }
            outputFile = outputDir.getChildFile(newZpkFileName);
            try (RandomAccessOutputFile out = RandomAccessOutputFile.from(outputFile);
                 RandomAccessInputFile inMatch = RandomAccessInputFile.from(zpkFile.file)) {
                return patch(out, in, inMatch, ByteBuffer.wrap(indexEntry.getValue()));
            }
        } // try
    }

    public static void printUsage() {
        System.out.println("  java -jar gamepack.jar zpk patch -z <zpk_file> -i <input_diff> [-o <output_dir>]");
        System.out.println("  java -jar gamepack.jar zpk diff -z <zpk_file> -i <input_dir> [-o <output_dir>]"
                + " -E <exclude_pattern> -E <exclude_pattern> ...");
        System.out.println("  java -jar gamepack.jar zpk diff -z <zpk_file> -i <input_dir> [-o <output_dir>]"
                + " -I <include_pattern> -I <include_pattern> ...");
    }

    public static void usage() {
        System.out.println("Usage:");
        printUsage();
        System.exit(1);
    }

    public static void callMain(String[] args) throws IOException {
        AbstractFile zFile = null;
        AbstractFile outputDir = null;
        AbstractFile inputFile = null;
        List<String> filterList = new ArrayList<>();
        int includeOrExclude = 0;
        if (args.length < 6) {
            usage();
            return;
        }
        final String op = args[1];
        int i = 2;
        while (i < args.length) {
            switch (args[i++]) {
                case "-z" -> {
                    if (i < args.length) {
                        zFile = AbstractFile.make(args[i++]);
                    }
                }
                case "-i" -> {
                    if (i < args.length) {
                        inputFile = AbstractFile.make(args[i++]);
                    }
                }
                case "-o" -> {
                    if (i < args.length) {
                        outputDir = AbstractFile.make(args[i++]);
                    }
                }
                case "-E" -> {
                    if (i < args.length && includeOrExclude != 1) {
                        includeOrExclude = 2;
                        filterList.add(args[i++]);
                    }
                }
                case "-I" -> {
                    if (i < args.length) {
                        includeOrExclude = 1;
                        filterList.add(args[i++]);
                    }
                }
                case "-d" -> ENABLE_DEBUG = true;
                default -> usage();
            }
        }
        if (null == zFile || null == inputFile) {
            usage();
            return;
        }
        String name = zFile.getName();
        if (name.length() < 21) {
            throw new IOException("invalid zpk file name");
        }
        long bytesWritten = 0;
        ZpkFileDiff builder = new ZpkFileDiff();
        final long buildTime = builder.buildTime;
        switch (op) {
            case "patch" -> bytesWritten = builder.patch(outputDir, zFile, inputFile);
            case "diff" -> bytesWritten = builder.diff(outputDir, zFile, inputFile,
                    includeOrExclude == 1 ? new IncludesEnumFilter(filterList) :
                    includeOrExclude == 2 ? new ExcludesEnumFilter(filterList) : null);
            default -> {
                usage();
                return;
            }
        }
        final long cost = System.currentTimeMillis() - buildTime;
        System.out.println("Done! ");
        System.out.println("Output File: " + builder.outputFile + ". "
                + "Size: " + humanReadableBytes(bytesWritten) + ". "
                + "Speed: " + humanReadableBytes(1000.0 * bytesWritten / cost) + "/s" + ". "
                + "Cost: " + humanReadableTime(cost));
        System.out.println();
        System.exit(0);
    } // callMain
} // class ZpkFileDiff
